import asyncio
import importlib
import time
from datetime import datetime
from py_pli.pylib import EndPointType
from py_pli.pylib import URPCFunctions
from urpc_enum.moverparameter import MoverParameter
from urpc_enum.corexymoverparameter import CoreXYMoverParameter
from urpc_enum.measurementparameter import MeasurementParameter
from virtualunits.meas_seq_generator import meas_seq_generator


# From the terminal use the names defined in this function to access the endpoints directly.
def __getattr__(name):
    if name in nodes:
        return get_node_endpoint(name)
    if name in movers:
        return get_mover_endpoint(name)
    if name == 'corexy':
        return get_corexy_endpoint()
    if name == 'measurement' or name == 'meas':
        return get_measurement_endpoint()
    if name in serial_endpoints:
        return get_serial_endpoint(name)
    if name == 'sys':
        return get_system_control_endpoint()
    if name == 'fan':
        return get_fan_control_endpoint()
    if name in temperature_controls:
        return get_temperature_control_endpoint(name)
    raise AttributeError(f"module '{__name__}' has no attribute '{name}'")


# Helper function to create endpoints that are not supported by the PyRunner yet.
def create_endpoint(can_id, endpoint_class_name):
    endpoint_creator = URPCFunctions.instance.endPointCreator

    endpoint_module = importlib.import_module('urpc.' + endpoint_class_name.lower())
    endpoint_class = getattr(endpoint_module, endpoint_class_name)

    endpoint = endpoint_class(can_id, endpoint_creator.transmitterQueueParent, endpoint_creator.receptionQueueChild, endpoint_creator.iso15765xSend, endpoint_creator.event_loop)

    endpoint_creator.creationEvent.clear()
    endpoint_creator.createEndpointQueueParent.send(can_id)
    endpoint_creator.creationEvent.wait()
    endpoint_creator.creationEvent.clear()

    endpoint_creator.urpcFunctions.endPointsDic[can_id] = endpoint

    return endpoint


# Float to Q31 format.
def float_to_q31_old(value):
    q31 = round(value * 2**31)
    if q31 > 2**31 - 1:
        q31 = 2**31 - 1
    if q31 < -2**31:
        q31 = -2**31
    return int(q31)


# Q31 format to Float.
def q31_to_float_old(q31):
    value = q31 / 2**31
    return value

# Note: These two functions are redefined in line 664, so I had to rename them here for the master build to pass. ~Ruediger B.

# Node Endpoint ########################################################################################################

nodes = {
    'eef'       : {'id':0x0008, 'delay':5},
    'mc6'       : {'id':0x0010, 'delay':1},
    'mc6_stk'   : {'id':0x0028, 'delay':1},
    'fmb'       : {'id':0x0018, 'delay':1},
    'pmc'       : {'id':0x00F0, 'delay':1},
    'pmc1'      : {'id':0x00F1, 'delay':1},
    'pmc2'      : {'id':0x00F2, 'delay':1},
    'pmc3'      : {'id':0x00F3, 'delay':1},
}

def get_node_endpoint(node_name):
    can_id = nodes[node_name]['id']
    node = URPCFunctions.instance.endPointsDic.get(can_id)
    if node is None:
        node = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, EndPointType.NODE)
    return node

async def start_firmware(node_name):
    node = get_node_endpoint(node_name)
    if (await node.GetFirmwareInfo(timeout=1))[0] != 1:
        await node.StartFirmware(timeout=1)
        await asyncio.sleep(nodes[node_name]['delay'])
        if (await node.GetFirmwareInfo(timeout=1))[0] != 1:
            raise Exception(f"Failed to start the firmware.")


async def get_din(node_name):
    await start_firmware(node_name)
    node = get_node_endpoint(node_name)
    din = (await node.GetDigitalInput(1))[0]
    await asyncio.sleep(0.5)
    print(f"      21098765432109876543210987654321")
    print(f"din = {din:032b}")


# Mover Endpoint ########################################################################################################

movers = {
    'as1'       : {'id':0x010A},  # Aperture Slider 1
    'as2'       : {'id':0x010B},  # Aperture Slider 2
    'fms'       : {'id':0x0110},  # Filder Module Slider
    'usfm'      : {'id':0x0111},  # US Lum Focus Mover
    'fm'        : {'id':0x0112},  # Focus Mover
    'bld'       : {'id':0x0113},  # Bottom Light Director
    'pd'        : {'id':0x0114},  # Plate Door
    'els'       : {'id':0x0115},  # Excitation Light Selector
    'zdrive'    : {'id':0x01F0},  # Z-Drive of PMC
    'pipettor'  : {'id':0x02F0},  # Pipettor of PMC

# Alternative Naming for MC6 Tests
    'mc6m1'     : {'id':0x0110},  # Filder Module Slider
    'mc6m2'     : {'id':0x0111},  # US Lum Focus Mover
    'mc6m3'     : {'id':0x0112},  # Focus Mover
    'mc6m4'     : {'id':0x0113},  # Bottom Light Director
    'mc6m5'     : {'id':0x0114},  # Plate Door
    'mc6m6'     : {'id':0x0115},  # Excitation Light Selector

# Alternative Naming for MC6_STK Tests
    'mc6stkm1'     : {'id':0x0128},  # Lift Mover Left
    'mc6stkm2'     : {'id':0x0129},  # Lift Mover Right
    'mc6stkm3'     : {'id':0x012A},  # Release Mover Left
    'mc6stkm4'     : {'id':0x012B},  # Release Mover Right
    'mc6stkm5'     : {'id':0x012C},  # No Function
    'mc6stkm6'     : {'id':0x012D},  # No Function
}

def get_mover_endpoint(mover_name):
    can_id = movers[mover_name]['id']
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, EndPointType.MOVER)
    return endpoint


# CoreXY Endpoint ######################################################################################################

def get_corexy_endpoint():
    can_id = 0x0108
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, EndPointType.COREXY)
    return endpoint


# Measurement Endpoint #################################################################################################

def get_measurement_endpoint():
    can_id = 0x010C
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, EndPointType.MeaurementFunctions)
    return endpoint


# Serial Endpoint ######################################################################################################

serial_endpoints = {
    'trf'   : {'id':0x010D, 'node':'eef', 'ntrig':3},
    'bcr1'  : {'id':0x010E, 'node':'eef', 'ntrig':3},
    'bcr2'  : {'id':0x010F, 'node':'eef', 'ntrig':4},
    'bcr3'  : {'id':0x0208, 'node':'eef', 'ntrig':5, 'mux':0},
    'bcr4'  : {'id':0x0208, 'node':'eef', 'ntrig':5, 'mux':1},
    'bcr5'  : {'id':0x0208, 'node':'eef', 'ntrig':5, 'mux':2},
}

def get_serial_endpoint(serial_endpoint_name):
    can_id = serial_endpoints[serial_endpoint_name]['id']
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, EndPointType.SERIAL)
    return endpoint


# System Control Endpoint ##############################################################################################

def get_system_control_endpoint():
    can_id = 0x0120
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = create_endpoint(can_id, 'SystemControlFunctions')
    return endpoint


# Fan Control Endpoint #################################################################################################

def get_fan_control_endpoint():
    can_id = 0x0121
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = create_endpoint(can_id, 'FanControlFunctions')
    return endpoint


# Temperature Control Endpoint #########################################################################################

temperature_controls = {
    'eef_tec'    : {'id':0x0209},
    'fmb_tec'    : {'id':0x0122},
}

def get_temperature_control_endpoint(control_name):
    can_id = temperature_controls[control_name]['id']
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = create_endpoint(can_id, 'TemperatureControlFunctions')
    return endpoint


# Initialization #######################################################################################################

async def init(*endpoints):
    for endpoint in endpoints:
        if endpoint in nodes:
            await start_firmware(endpoint)
        if endpoint in movers:
            mover = get_mover_endpoint(endpoint)
            if endpoint == 'as1' or endpoint == 'as2':
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=14, holdPower=1, drivePowerHoldTime=1000, drivePowerFallTime=1000, timeout=1
                )
                await mover.UseProfile(0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000, timeout=1)
                await mover.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01, timeout=1)
                await mover.SetParameter(MoverParameter.MovementDirection,                 1, timeout=1)
                await mover.SetConfigurationStatus(1, timeout=1)
            elif endpoint == 'fm':
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=10, holdPower=1, drivePowerHoldTime=1000, drivePowerFallTime=1000, timeout=1
                )
                await mover.UseProfile(0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000, timeout=1)
                await mover.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01, timeout=1)
                await mover.SetParameter(MoverParameter.MovementDirection,                 0, timeout=1)
                await mover.SetConfigurationStatus(1, timeout=1)
            else:
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000, timeout=1
                )
                await mover.UseProfile(0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000, timeout=1)
                await mover.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01, timeout=1)
                await mover.SetParameter(MoverParameter.MovementDirection,                 1, timeout=1)
                await mover.SetConfigurationStatus(1, timeout=1)
        if endpoint == 'corexy':
            corexy = get_corexy_endpoint()
            await corexy.SetProfile(
                handle=0, speed=100000, accel=2000000, decel=2000000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000, timeout=1
            )
            await corexy.UseProfile(0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.RampMode,                           1, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeAxisOrder,                      1, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSearchDirectionX,               0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSearchDirectionY,               0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxDistanceX,              550000, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxDistanceY,              550000, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxReverseDistanceX,        10000, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxReverseDistanceY,        10000, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeExtraReverseDistanceX,          0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeExtraReverseDistanceY,          0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeCalibrationSpeed,           10000, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomePositionX,             0x7FFFFFFF, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomePositionY,             0x7FFFFFFF, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSensorEnableX,               0x01, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSensorEnableY,               0x01, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.EncoderMode,                        1, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.EncoderCorrectionFactorX,       12800, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.EncoderCorrectionFactorY,       12800, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.MaxEncoderDeviation,              512, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.MovementDirectionX,                 0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.MovementDirectionY,                 0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.RotationalDirectionA,               0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.RotationalDirectionB,               0, timeout=1)
            await corexy.SetConfigurationStatus(1, timeout=1)
        if endpoint == 'alpha_tec':
            tec = get_temperature_control_endpoint('eef_tec')
            reference = float_to_q31(0.33)
            await tec.SetParameter(channel=0, number=0, value=reference, timeout=1)
            await tec.SelectAnalogInput(channel=0, number=3, scaling=1.0, timeout=1)
            await tec.SelectAnalogOutput(channel=0, number=11, scaling=1.0, timeout=1)
            await tec.Configure(channel=0, dt=0.01, kp=-10.0, ti=float('inf'), td=0.0, min=0.0, max=+0.15, timeout=1)
            await tec.Enable(channel=0, enable=1, timeout=1)

        await asyncio.sleep(0.5)
        print(f"{endpoint} initialized")


# Mover Tests ##########################################################################################################

async def fragmented_move(mover_name, position, step_size=1, timeout=1):
    mover = get_mover_endpoint(mover_name)
    current_position = (await mover.GetPosition())[0]
    print(f"current_position = {current_position}")
    if (position >= current_position):
        dir = +1
    else:
        dir = -1
    for pos in range((current_position + (step_size * dir)), (position + dir), (step_size * dir)):
        print(f"Move({pos})")
        await mover.Move(pos, timeout)


# Sequencer Tests ######################################################################################################

signals = {
    'flash'     : (1 << 23),
    'alpha'     : (1 << 22),
    'ingate2'   : (1 << 17),
    'ingate1'   : (1 << 16),
    'hvgate2'   : (1 << 13),
    'hvgate1'   : (1 << 12),
    'hvon3'     : (1 << 10),
    'hvon2'     : (1 <<  9),
    'hvon1'     : (1 <<  8),
    'rstaux'    : (1 <<  6),
    'rstabs'    : (1 <<  5),
    'rstref'    : (1 <<  4),
    'rstpmt2'   : (1 <<  1),
    'rstpmt1'   : (1 <<  0),
}


triggers = {
    'trf'   : (1 << 8),
    'aux'   : (1 << 6),
    'abs'   : (1 << 5),
    'ref'   : (1 << 4),
    'pmt3'  : (1 << 2),
    'pmt2'  : (1 << 1),
    'pmt1'  : (1 << 0),
}

channels = {
    'pmt1'  : (0 << 24),
    'pmt2'  : (1 << 24),
    'pmt3'  : (2 << 24),
    'pmt4'  : (3 << 24),
    'ref'   : (4 << 24),
    'abs'   : (5 << 24),
    'res'   : (6 << 24),
}


async def seq_pulse_signal(name, duration_us=1):
    meas = get_measurement_endpoint()
    sequence = [
        0x7C000000 | (duration_us * 100 - 1),
        0x02000000 | signals[name],
        0x7C000000,
        0x03000000 | signals[name],
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    await asyncio.sleep(0.5)


async def seq_set_signal(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x02000000 | signals[name],
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    await asyncio.sleep(0.5)


async def seq_clear_signal(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x03000000 | signals[name],
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    await asyncio.sleep(0.5)


async def seq_trigger(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x01000000 | triggers[name],
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    await asyncio.sleep(0.5)


async def seq_wait_for_trigger():
    meas = get_measurement_endpoint()
    sequence = [
        0x74000000 | 1,
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    done = False
    while not done:
        print(f"waiting for trigger...")
        status = (await meas.GetStatus())[0]
        if status & 0x01:
            done = True
        else:
            await asyncio.sleep(0.1)
    await asyncio.sleep(0.5)
    print(f"waiting for trigger done")


async def seq_get_analog(detector='pmt1'):
    meas = get_measurement_endpoint()
    sequence = [
        0x8C040000,                         # Clear Result Buffer
        0x01000000 | triggers[detector],    # Trigger Analog Measurement
        0x80000000 | channels[detector],    # Get Analog High Result
        0x80100001 | channels[detector],    # Get Analog Low Result
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    results = await meas.ReadResults(0, 4, timeout=5)
    await asyncio.sleep(0.5)
    
    print(f"Analog Low:  {detector} = {5 / 65536 * results[0]} V")
    print(f"Analog High: {detector} = {5 / 65536 * results[1]} V")


async def seq_get_counter(detector='pmt1', window_us=100):
    meas = get_measurement_endpoint()
    window_corse, window_fine = divmod(window_us, 65536)
    sequence = [
        0x7C000000 | (100 - 1),
        0x88B80000 | channels[detector],
    ]
    if window_corse > 0:
        sequence.extend([
            0x07000000 | (window_corse - 1),
            0x07000000 | (65536 - 1),
            0x7C000000 | (100 - 1),
            0x88D00000 | channels[detector],
            0x05000000,
            0x05000000,
        ])
    if window_fine > 0:
        sequence.extend([
            0x07000000 | (window_fine - 1),
            0x7C000000 | (100 - 1),
            0x88D00000 | channels[detector],
            0x05000000,
        ])
    sequence.extend([
        0x80900000 | channels[detector],
        0x00000000,
    ])
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    results = await meas.ReadResults(0, 1, timeout=5)
    await asyncio.sleep(0.5)
    
    print(f"Count: {detector} = {results[0]}")


async def seq_measure(window_us=1000):
    meas = get_measurement_endpoint()
    window_corse, window_fine = divmod(window_us, 65536)
    gate_delay = 1000000  # 10 ms
    sequence = [
        0x7C000000 | (gate_delay - 1),
        0x02000000 | signals['hvgate1'] | signals['hvgate2'] | signals['ingate1'] | signals['ingate2'] | signals['rstpmt1'] | signals['rstpmt2'],
        0x8C000000,
        0x8C000001,
        0x8C000002,
        0x8C000003,
        0x8C000004,
        0x8C000005,
        0x7C000000 | (100 - 1),
        0x03000000 | signals['rstpmt1'] | signals['rstpmt2'],
        0x88B80000 | channels['pmt1'],
        0x88B80000 | channels['pmt2'],
    ]
    if window_corse > 0:
        sequence.extend([
            0x07000000 | (window_corse - 1),
            0x07000000 | (65536 - 1),
            0x7C000000 | (100 - 1),
            0x88D00000 | channels['pmt1'],
            0x88D00000 | channels['pmt2'],
            0x05000000,
            0x05000000,
        ])
    if window_fine > 0:
        sequence.extend([
            0x07000000 | (window_fine - 1),
            0x7C000000 | (100 - 1),
            0x88D00000 | channels['pmt1'],
            0x88D00000 | channels['pmt2'],
            0x05000000,
        ])
    sequence.extend([
        0x01000000 | triggers['pmt1'] | triggers['pmt2'],
        0x80900000 | channels['pmt1'],
        0x80000001 | channels['pmt1'],
        0x80100002 | channels['pmt1'],
        0x80900003 | channels['pmt2'],
        0x80000004 | channels['pmt2'],
        0x80100005 | channels['pmt2'],
        0x03000000 | signals['hvgate1'] | signals['hvgate2'] | signals['ingate1'] | signals['ingate2'],
        0x00000000,
    ])
    for i in range(0, len(sequence), 28):
        subsequence = sequence[i:(i + 28)]
        buffer = [0] * 28
        buffer[0:len(subsequence)] = subsequence
        await meas.WriteSequence(i, len(subsequence), buffer, timeout=5)

    await meas.StartSequence(0, timeout=1)
    results = await meas.ReadResults(0, 6, timeout=(2 + window_us / 1000000))
    await asyncio.sleep(0.5)
    
    print(f"Count       pmt1 = {results[0]}")
    print(f"Analog Low  pmt1 = {5 / 65536 * results[1]} V")
    print(f"Analog High pmt1 = {5 / 65536 * results[2]} V")
    print(f"Count       pmt2 = {results[3]}")
    print(f"Analog Low  pmt2 = {5 / 65536 * results[4]} V")
    print(f"Analog High pmt2 = {5 / 65536 * results[5]} V")

async def seq_measure_no_analog(window_us=1000):
    meas = get_measurement_endpoint()
    window_corse, window_fine = divmod(window_us, 65536)
    gate_delay = 1000000  # 10 ms
    sequence = [
        0x7C000000 | (gate_delay - 1),
        0x02000000 | signals['hvgate1'] | signals['hvgate2']  | signals['rstpmt1'] | signals['rstpmt2'],
        0x03000000 | signals['ingate1'] | signals['ingate2'],
        0x8C000000,
        0x8C000001,
        0x8C000002,
        0x8C000003,
        0x8C000004,
        0x8C000005,
        0x7C000000 | (100 - 1),
        0x03000000 | signals['rstpmt1'] | signals['rstpmt2'],
        0x88B80000 | channels['pmt1'],
        0x88B80000 | channels['pmt2'],
    ]
    if window_corse > 0:
        sequence.extend([
            0x07000000 | (window_corse - 1),
            0x07000000 | (65536 - 1),
            0x7C000000 | (100 - 1),
            0x88D00000 | channels['pmt1'],
            0x88D00000 | channels['pmt2'],
            0x05000000,
            0x05000000,
        ])
    if window_fine > 0:
        sequence.extend([
            0x07000000 | (window_fine - 1),
            0x7C000000 | (100 - 1),
            0x88D00000 | channels['pmt1'],
            0x88D00000 | channels['pmt2'],
            0x05000000,
        ])
    sequence.extend([
        0x01000000 | triggers['pmt1'] | triggers['pmt2'],
        0x80900000 | channels['pmt1'],
        0x80000001 | channels['pmt1'],
        0x80100002 | channels['pmt1'],
        0x80900003 | channels['pmt2'],
        0x80000004 | channels['pmt2'],
        0x80100005 | channels['pmt2'],
        0x03000000 | signals['hvgate1'] | signals['hvgate2'] | signals['ingate1'] | signals['ingate2'],
        0x00000000,
    ])
    for i in range(0, len(sequence), 28):
        subsequence = sequence[i:(i + 28)]
        buffer = [0] * 28
        buffer[0:len(subsequence)] = subsequence
        await meas.WriteSequence(i, len(subsequence), buffer, timeout=5)

    await meas.StartSequence(0, timeout=1)
    results = await meas.ReadResults(0, 6, timeout=(2 + window_us / 1000000))
    await asyncio.sleep(0.5)
    
    print(f"Count       pmt1 = {results[0]}")

    print(f"Count       pmt2 = {results[3]}")
    

async def seq_darkcount(iterations, delay=100000):
    meas = get_measurement_endpoint()
    sequence = [
        0x7C000000 | (delay - 1),           # TimerWaitAndRestart(delay - 1)
        0x88B80000,                         # PulseCounterControl(ch=0, add=0, RstCnt=1, RstPreCnt=1, corr=1)
        0x07000000 | (iterations - 1),      # Loop(iterations - 1)
        0x7C000000 | (delay - 1),           #     TimerWaitAndRestart(delay - 1)
        0x88D00000,                         #     PulseCounterControl(ch=0, add=1, RstCnt=0, RstPreCnt=1, corr=0)
        0x05000000,                         # LoopEnd()
        0x80900000,                         # GetPulseCounterResult(ch=0, rel=0, RstCnt=1, add=0, dword=0, addrReg=0, addr=0)
        0x00000000,                         # Stop(0)
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    results = await meas.ReadResults(0, 1, timeout=5)
    await asyncio.sleep(0.5)
    
    print(f"Dark Count = {results[0]}")
    
    
async def set_pmt_hv(pmt, value):
    meas = get_measurement_endpoint()
    if pmt == 'pmt1' or pmt == 1:
        await meas.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, float_to_q31(value),timeout=1)
    if pmt == 'pmt2' or pmt == 2:
        await meas.SetParameter(MeasurementParameter.PMT2HighVoltageSetting, float_to_q31(value),timeout=1)
    if pmt == 'pmtus' or pmt == 'pmt3' or pmt == 3:
        await meas.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageSetting, float_to_q31(value),timeout=1)

async def set_pmt_dl(pmt, value):
    meas = get_measurement_endpoint()
    if pmt == 'pmt1' or pmt == 1:
        await meas.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, float_to_q31(value),timeout=1)
    if pmt == 'pmt2' or pmt == 2:
        await meas.SetParameter(MeasurementParameter.PMT2DiscriminatorLevel, float_to_q31(value),timeout=1)
    if pmt == 'pmtus' or pmt == 'pmt3' or pmt == 3:
        await meas.SetParameter(MeasurementParameter.PMTUSLUMDiscriminatorLevel, float_to_q31(value),timeout=1)

        
# Float to Q31 format.
def float_to_q31(value):
    if value < -1.0 or value > +1.0:
        raise ValueError(f"value must be in the range [+1.0, -1.0]")
    q31 = round(value * 2**31)
    if q31 > 2**31 - 1:
        q31 = 2**31 - 1
    if q31 < -2**31:
        q31 = -2**31
    return int(q31)


# Q31 format to Float.
def q31_to_float(q31):
    value = q31 / 2**31
    return value

async def set_aout(node_name, output_number, value):
    node = get_node_endpoint(node_name)
    await node.SetAnalogOutput(output_number, float_to_q31(value), timeout=1)
    return f"set_analog_output() done"


async def get_ain(node_name, input_number, value):
    node = get_node_endpoint(node_name)
    q31 = (await node.GetAnalogInput(input_number, timeout=1))[0]
    return q31_to_float(q31)



async def trigger_n_times(frequency, iterations, duration_us, flash_pwr=0.0, high_pwr=0):

    meas = get_measurement_endpoint()
    
    # Cancel if HighPower is set and Freqency above 500
    if frequency > 500 and high_pwr == 1:
        raise Exception(f"No HighPwr above 500Hz")
    
    node = get_node_endpoint('eef')
    
    # Set the Flash_Pwr if given
    if flash_pwr is not None:
        await set_aout('eef', 12, flash_pwr)
        await asyncio.sleep(0.1)
    
    # Set HighPower 
    await node.SetDigitalOutput(9,high_pwr)
    await asyncio.sleep(0.1)


    ontime = int(100 * duration_us)
    offtime = int(1 / frequency * 1e8 - ontime)
    loop_a = int(iterations / 65536)
    loop_b = int(iterations % 65536)
    data = []
    if loop_a > 0:
        data.extend([
            0x07000000 | (loop_a - 1),      # Loop A Multiple of 2^16
            0x07000000 | (65536 - 1),       # Loop Inner
            0x7C000000 | (ontime - 1),      # Wait for offtime timer then start ontime timer
            0x02000000 | signals['flash'],  # Set Flash_Trg high
            0x7C000000 | (offtime - 1),     # Wait for ontime timer then start offtime timer
            0x03000000 | signals['flash'],  # Set Flash_Trg low
            0x05000000,                     # LoopEnd
            0x05000000,                     # LoopEnd
        ])
    if loop_b > 0:
        data.extend([
            0x07000000 | (loop_b - 1),      # Loop B Remainder
            0x7C000000 | (ontime - 1),      # Wait for offtime timer then start ontime timer
            0x02000000 | signals['flash'],  # Set Flash_Trg high
            0x7C000000 | (offtime - 1),     # Wait for ontime timer then start offtime timer
            0x03000000 | signals['flash'],  # Set Flash_Trg low
            0x05000000,                     # LoopEnd
        ])
    data.extend([
        0x00000000,                     # Stop
    ])
    buffer = [0] * 28
    buffer[0:len(data)] = data
    await meas.WriteSequence(0, len(data), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    
async def new_eef_old_fdb(frequency, iterations, duration_us, flash_pwr=None, high_pwr=0):

    meas = get_measurement_endpoint()
    
    # Cancel if HighPower is set and Freqency above 500
    if frequency > 500 and high_pwr == 1:
        raise Exception(f"No HighPwr above 500Hz")
    
    node = get_node_endpoint('eef')
    
    # Set the Flash_Pwr if given
    if flash_pwr is not None:
        await node.SetAnalogOutput(2, int(flash_pwr))
        await asyncio.sleep(0.1)
    
    # Set HighPower 
    if high_pwr == 1:
        await node.SetDigitalOutput(9,0) #caution -> inverted
        await asyncio.sleep(0.1)
    else:
        await node.SetDigitalOutput(9,1) #caution -> inverted 
        await asyncio.sleep(0.1)
    
    ontime = int(100 * duration_us)
    offtime = int(1 / frequency * 1e8 - ontime)
    loop_a = int(iterations / 65536)
    loop_b = int(iterations % 65536)
    data = []
    if loop_a > 0:
        data.extend([
            0x07000000 | (loop_a - 1),      # Loop A Multiple of 2^16
            0x07000000 | (65536 - 1),       # Loop Inner
            0x7C000000 | (ontime - 1),      # Wait for offtime timer then start ontime timer
            0x02000000 | signals['flash'],  # Set Flash_Trg high
            0x7C000000 | (offtime - 1),     # Wait for ontime timer then start offtime timer
            0x03000000 | signals['flash'],  # Set Flash_Trg low
            0x05000000,                     # LoopEnd
            0x05000000,                     # LoopEnd
        ])
    if loop_b > 0:
        data.extend([
            0x07000000 | (loop_b - 1),      # Loop B Remainder
            0x7C000000 | (ontime - 1),      # Wait for offtime timer then start ontime timer
            0x02000000 | signals['flash'],  # Set Flash_Trg high
            0x7C000000 | (offtime - 1),     # Wait for ontime timer then start offtime timer
            0x03000000 | signals['flash'],  # Set Flash_Trg low
            0x05000000,                     # LoopEnd
        ])
    data.extend([
        0x00000000,                     # Stop
    ])
    buffer = [0] * 28
    buffer[0:len(data)] = data
    await meas.WriteSequence(0, len(data), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)

# Serial Tests #########################################################################################################

async def serial_loopback(serial_name, *txdata):
    serial = get_serial_endpoint(serial_name)
    await serial.SetParameter(0, 1900)
    await serial.ClearReadBuffer()
    await serial.Write(len(txdata), bytes(txdata), timeout=2)
    rxdata = None
    try:
        response = await serial.Read(len(txdata), timeout=2)
        length = response[0]
        rxdata = response[1:(1 + length)]
    except BaseException as ex:
        print(ex)
    await asyncio.sleep(0.5)
    print(f"rxdata: {rxdata}")
    
    



# BCR Tests ############################################################################################################

async def bcr_write(bcr_name, *data):
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(6, bool(mux & 1))
        await node.SetDigitalOutput(7, bool(mux & 2))
        await node.SetDigitalOutput(8, bool(mux & 4))
    await bcr.Write(len(data), bytes(data), timeout=2)
    await asyncio.sleep(0.5)


async def bcr_read(bcr_name, length, timeout=5, sw_trigger=False):
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    ntrig = serial_endpoints[bcr_name]['ntrig']
    barcode = None
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(6, bool(mux & 1))
        await node.SetDigitalOutput(7, bool(mux & 2))
        await node.SetDigitalOutput(8, bool(mux & 4))
    await bcr.SetParameter(0, int(timeout * 1000))
    await bcr.ClearReadBuffer()
    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 0)
    else:
        await bcr.Write(2, b'\x1B\x31', timeout=2)
    try:
        response = await bcr.Read(length, timeout=(timeout + 0.1))
        length = response[0]
        barcode = response[1:(1 + length)]
    except BaseException as ex:
        print(ex)
    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 1)
    await asyncio.sleep(0.5)
    print(f"Barcode: {barcode}")


async def bcr_read_until(bcr_name, termination='\r', timeout=5, sw_trigger=False):
    if isinstance(termination, str):
        termination = ord(termination)
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    ntrig = serial_endpoints[bcr_name]['ntrig']
    barcode = None
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(6, bool(mux & 1))
        await node.SetDigitalOutput(7, bool(mux & 2))
        await node.SetDigitalOutput(8, bool(mux & 4))
    await bcr.SetParameter(0, (timeout * 1000))
    await bcr.ClearReadBuffer()
    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 0)
    else:
        await bcr.Write(2, b'\x1B\x31', timeout=2)
    try:
        response = await bcr.ReadUntil(termination, timeout=(timeout + 0.1))
        length = response[0]
        barcode = response[1:(1 + length)]
    except BaseException as ex:
        print(ex)
    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 1)
    await asyncio.sleep(0.5)
    print(f"Barcode: {barcode}")


async def bcr_overflow(bcr_name):
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    ntrig = serial_endpoints[bcr_name]['ntrig']
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(6, bool(mux & 1))
        await node.SetDigitalOutput(7, bool(mux & 2))
        await node.SetDigitalOutput(8, bool(mux & 4))
    await bcr.ClearReadBuffer()
    for i in range(100):
        await node.SetDigitalOutput(ntrig, 0)
        await asyncio.sleep(0.9)
        await node.SetDigitalOutput(ntrig, 1)
        await asyncio.sleep(0.1)
    await asyncio.sleep(0.5)
    print(f"BCR Status: {await bcr.GetStatus()}")


# Temperature Control Tests ############################################################################################

async def tec_log(name, channel, interval, duration):
    tec = get_temperature_control_endpoint(name)
    file = open('temperature_control_log.txt', 'a')
    # file.write(f"TIMESTAMP               ; REFERENCE[FS] ; FEEDBACK[FS]  ; OUTPUT[FS]\n")
    file.write(f"TIMESTAMP[s]  ; REFERENCE[FS] ; FEEDBACK[FS]  ; OUTPUT[FS]\n")
    start = time.perf_counter()
    timestamp = start
    while (timestamp - start) < duration:
        results = await asyncio.gather(
            tec.GetParameter(channel, 0, timeout=1),
            tec.GetFeedbackValue(channel, timeout=1),
            tec.GetOutputValue(channel, timeout=1),
            asyncio.sleep(interval)
        )
        reference = q31_to_float(results[0][0])
        feedback  = q31_to_float(results[1][0])
        output    = q31_to_float(results[2][0])
        # file.write(f"{datetime.now().isoformat(timespec='milliseconds')} ; {reference:+13.6f} ; {feedback:+13.6f} ; {output:+13.6f}\n")
        file.write(f"{timestamp:13.3f} ; {reference:+13.6f} ; {feedback:+13.6f} ; {output:+13.6f}\n")
        timestamp = time.perf_counter()
    
    file.write(f"\n")
    file.close()


async def tec_alpha_test():
    eef = get_node_endpoint('eef')
    tec = get_temperature_control_endpoint('eef_tec')

    await init('eef', 'alpha_tec')              # Initialize EEF and Alpha TEC
    await eef.SetAnalogOutput(0, 1626881551)    # Set Laser Diode Current (≈2,5V)
    await eef.SetDigitalOutput(0, 1)            # Enable laser diode current supply
    await seq_set_signal('alpha')               # Disable alpha laser
    await asyncio.sleep(10)                     # Wait 10 s to stabalize the temperature
    await seq_clear_signal('alpha')             # Enable alpha laser
    await asyncio.sleep(20)                     # Wait 10 s to stabalize the temperature
    await seq_set_signal('alpha')               # Disable alpha laser



async def tec_alpha_test_log():
    await asyncio.gather(
        tec_log('eef_tec', 0, 0, 50),
        tec_alpha_test()
    )

# MC6 Test Scripts for initial operation

async def test_mover(Mover):
    movstr = 'mc6' + Mover
    mov = get_mover_endpoint(movstr)

    await init('mc6', movstr)
    await mov.Move(50000)
    await asyncio.sleep(1)
    await mov.Move(25000)
    await asyncio.sleep(1)
    await mov.Home()

    print('\n'+'Test of ' + Mover + ' successful'+'\n')

async def test_ref(ref):
    mc6 = get_node_endpoint('mc6')
    x = 0
    await init('mc6')
    while x == 0:
           x = (await mc6.GetDigitalInput(ref-1))[0]

    print('\n'+'Test of REF' + str(ref-1) + ' successful'+'\n')

async def test_enc(enc):
    mc6 = get_node_endpoint('mc6')
    x = 1
    y = 1
    z = 1
    temp1 = 0
    temp2 = 0
    temp3 = 0
    if enc == 1:
        temp1 = 12
        temp2 = 13
        temp3 = 14
    elif enc == 2:
        temp1 = 15
        temp2 = 16
        temp3 = 17
    else:
        print('\n'+ str(enc) + ' is no valid input (only 1 or 2)'+'\n')
    await init('mc6')
    while x == 1 or y == 1 or z == 1:
           x = (await mc6.GetDigitalInput(temp1))[0]
           y = (await mc6.GetDigitalInput(temp2))[0]
           z = (await mc6.GetDigitalInput(temp3))[0]

    print('\n'+'Test of ENC' + str(enc) + ' successful'+'\n')   
        
async def test_adc():
    mc6 = get_node_endpoint('mc6')

    await init('mc6')

    x = (await mc6.GetAnalogInput(2))[0]
    y = (await mc6.GetAnalogInput(3))[0]

    if x < 0.01:
        print('\n'+'ADC 1 --> '+str(x)+'\n')
        print('\n'+'ADC 1 --> OK'+'\n')

    if y > 0.99:
        print('\n'+'ADC 2 --> '+str(y)+'\n')
        print('\n'+'ADC 2 --> OK'+'\n')

async def test_can():
    mc6 = get_node_endpoint('mc6')
    await init('mc6')
    print('\n'+'ID 1 = '+str((await mc6.GetFirmwareInfo(1))[0])+'\n'+'ID 2 = '+str((await mc6.GetFirmwareInfo(1))[1])+'\n')

async def test_i2c():
    mc6 = get_node_endpoint('mc6')

    await init('mc6')

    print('\n'+'Temperature Sensor Readout: ' + str(256*(await mc6.GetAnalogInput(7))[0]) +'°C'+'\n'+ 'I2C Test succesful!'+'\n')  

async def test_pwm(pwm):
    mc6 = get_node_endpoint('mc6')

    await init('mc6')

    await mc6.ConfigurePWMOutput(pwm,1000,100,10)

    await mc6.EnablePWMOutput(pwm,1)

